<?php

/**
 * Template tags and helper functions
 *
 * This file contains template tags and utility functions that can be used
 * throughout the theme to display common elements like post meta, navigation,
 * and other template-related functionality.
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Get template from template-parts folder
 *
 * A wrapper function for get_template_part() that provides a more consistent
 * way to load template parts with optional parameters.
 *
 * @param string $template The template part to load (without .php extension)
 * @param string $name     Optional. The name of the specialized template.
 * @param array  $args     Optional. Additional arguments to pass to the template.
 * @return void
 */
function rt_get_template_part( $template, $name = '', $args = '' ) {
	get_template_part( "template-parts/{$template}", $args, $name );
}

/**
 * Get logo
 *
 * @param string $type
 * @return string
 */
function rt_get_logo( $type ) {
	$logo             = get_theme_mod( 'brand_logo_primary' );
	$logo_alternative = '';

	$logo_mobile             = wp_parse_args( get_theme_mod( 'brand_logo_mobile' ), $logo );
	$logo_mobile_alternative = '';

	$site_name = get_bloginfo( 'name' );
	$site_url  = site_url();

	if ( $type == 'desktop' ) {
		$html = "<a class='rt-logo' href='{$site_url}' rel='home'>";

		if ( ! empty( $logo ) ) {
			$html .= wp_sprintf( "<span class='rt-logo__primary'><img src='%s' class='site-logo-img' width='%d' height='%d' alt='%s'></span>", esc_url( $logo['url'] ), $logo['width'], $logo['height'], $site_name );
		}

		if ( ! empty( $logo_alternative ) ) {
			$html .= wp_sprintf( "<span class='rt-logo__alternative'><img src='%s' class='site-logo-img' width='%d' height='%d' alt='%s'></span>", esc_url( $logo_alternative['url'] ), $logo_alternative['width'], $logo_alternative['height'], $site_name );
		}

		$html .= '</a>';
	}

	if ( $type == 'mobile' ) {
		$html = "<a class='rt-logo-mobile' href='{$site_url}' rel='home'>";

		if ( ! empty( $logo_mobile ) ) {
			$html .= wp_sprintf( "<span class='rt-logo-mobile__primary'><img src='%s' class='site-logo-img' width='%d' height='%d' alt='%s'></span>", esc_url( $logo_mobile['url'] ), $logo_mobile['width'], $logo_mobile['height'], $site_name );
		}

		if ( ! empty( $logo_mobile_alternative ) ) {
			$html .= wp_sprintf( "<span class='rt-logo-mobile__alternative'><img src='%s' class='site-logo-img' width='%d' height='%d' alt='%s'></span>", esc_url( $logo_mobile_alternative['url'] ), $logo_mobile_alternative['width'], $logo_mobile_alternative['height'], $site_name );
		}

		$html .= '</a>';
	}

	return $html;
}

/**
 * Get page title
 *
 * @return string
 */
function rt_get_page_title() {

	$title = esc_html__( 'Archives', 'kubelio' );
	$desc  = '';

	if ( is_archive() ) {
		$title = get_the_archive_title();
		$desc  = get_the_archive_description();
	}

	if ( is_singular() ) {
		$title = get_the_title();
	}

	if ( is_post_type_archive() ) {
		$title = post_type_archive_title( '', false );
	}

	if ( rt_is_woocommerce( 'shop' ) ) {
		$title = woocommerce_page_title( false );
		$desc  = '';
	}

	if ( is_author() ) {
		$title = get_the_author();
	}

	if ( is_year() ) {
		$title = get_the_date( _x( 'Y', 'yearly archives date format' ) );
	}

	if ( is_month() ) {
		$title = get_the_date( _x( 'F Y', 'monthly archives date format' ) );
	}

	if ( is_day() ) {
		$title = get_the_date( _x( 'F j, Y', 'daily archives date format' ) );
	}

	if ( is_home() ) {
		$blog_id    = get_option( 'page_for_posts' );
		$blog_title = get_the_title( $blog_id );

		$title = ( $blog_id ) ? $blog_title : esc_html__( 'Newest Post', 'kubelio' );
	}

	if ( is_404() ) {
		$title = esc_html__( 'Page Not Found', 'kubelio' );
	}

	if ( is_search() ) {
		$title = sprintf( esc_html__( 'Search Results for: %s', 'kubelio' ), esc_html( get_search_query( false ) ) );
	}

	rt_get_template_part(
		'global/page-title',
		array(
			'title' => $title,
			'desc'  => $desc,
		)
	);
}

/**
 * Get breadcrumbs
 *
 * Show breadcrumbs on the page, if yostseo or rankmath is active this function will use their breadcrumbs
 * Otherwise it will use the default breadcrumbs

 * @param array $args
 * @return string
 */
function rt_get_breadrumbs( $args = array() ) {
	if ( get_theme_mod( 'breadcrumbs', true ) ) {

		$args['class'][] = 'rt-breadcrumb';
		$block_class     = implode( ' ', $args['class'] );

		ob_start();

		$html = "<div class='{$block_class}'><div class='rt-breadcrumb__wrapper'>";

		if ( function_exists( 'bcn_display' ) ) {
			$html .= bcn_display();
		} elseif ( function_exists( 'yoast_breadcrumb' ) && ! empty( get_option( 'wpseo_titles' )['breadcrumbs-enable'] ) ) {
			$html .= yoast_breadcrumb( '', '', false );
		} elseif ( function_exists( 'rank_math_the_breadcrumbs' ) && rank_math_get_breadcrumbs() ) {
			$html .= rank_math_get_breadcrumbs();
		} else {
			$separator = "<span class='rt-breadcrumb__separator'> / </span>";

			$home_title = esc_html__( 'Home', 'kubelio' );

			// If you have any custom post types with custom taxonomies, put the taxonomy name below (e.g. product_cat)
			$custom_taxonomy = 'product_cat';

			// Get the query & post information
			global $post, $wp_query;

			// Do not display on the homepage
			if ( ! is_front_page() ) {

				// Home page
				$html .= '<span class="item-home"><a class="bread-link bread-home" href="' . get_home_url() . '" title="' . $home_title . '">' . $home_title . '</a></span>';
				$html .= '<span class="separator separator-home"> ' . $separator . ' </span>';

				if ( is_home() ) {
					$blog_id = get_option( 'page_for_posts' );

					$blog_title = ( $blog_id ) ? get_the_title( $blog_id ) : esc_html__( 'Newest Post', 'kubelio' );

					$html .= '<span class="item-current item-archive"><span class="bread-current bread-archive">' . $blog_title . '</span></span>';
				} elseif ( is_single() ) {
					// If post is a custom post type
					$post_type = get_post_type();

					// If it is a custom post type display name and link
					if ( $post_type != 'post' ) {
						$post_type_object  = get_post_type_object( $post_type );
						$post_type_archive = get_post_type_archive_link( $post_type );
						$post_type_name    = apply_filters( 'retheme_archive_title', $post_type_object->labels->name );

						$html .= '<span class="item-cat item-custom-post-type-' . $post_type . '"><a class="bread-cat bread-custom-post-type-' . $post_type . '" href="' . $post_type_archive . '" title="' . $post_type_name . '">' . $post_type_name . '</a></span>';
						$html .= '<span class="separator"> ' . $separator . ' </span>';
					}

					// Get post category info
					$category = get_the_category();

					if ( ! empty( $category ) ) {
						// Simpan hasil array_values ke dalam variabel
						$category_values = array_values( $category );

						// Get last category post is in
						$last_category = end( $category_values );

						// Get parent any categories and create array
						$get_cat_parents = rtrim( get_category_parents( $last_category->term_id, true, ',' ), ',' );
						$cat_parents     = explode( ',', $get_cat_parents );

						// Loop through parent categories and store in variable $cat_display
						$cat_display = '';
						foreach ( $cat_parents as $parents ) {
							$cat_display .= '<span class="item-cat">' . $parents . '</span>';
							$cat_display .= '<span class="separator"> ' . $separator . ' </span>';
						}
					}

					// If it's a custom post type within a custom taxonomy
					$taxonomy_exists = taxonomy_exists( $custom_taxonomy );

					if ( empty( $last_category ) && ! empty( $custom_taxonomy ) && $taxonomy_exists ) {
						$taxonomy_terms = get_the_terms( $post->ID, $custom_taxonomy );

						if ( $taxonomy_terms && ! is_wp_error( $taxonomy_terms ) ) {
							$cat_id       = $taxonomy_terms[0]->term_id;
							$cat_nicename = $taxonomy_terms[0]->slug;
							$cat_link     = get_term_link( $taxonomy_terms[0]->term_id, $custom_taxonomy );
							$cat_name     = $taxonomy_terms[0]->name;
						}
					}

					// Check if the post is in a category
					if ( ! empty( $last_category ) ) {
						$html .= $cat_display;
						$html .= '<span class="item-current item-' . $post->ID . '"><span class="bread-current bread-' . $post->ID . '" title="' . get_the_title() . '">' . get_the_title() . '</span></span>';
						// Else if post is in a custom taxonomy
					} elseif ( ! empty( $cat_id ) ) {
						$html .= '<span class="item-cat item-cat-' . $cat_id . ' item-cat-' . $cat_nicename . '"><a class="bread-cat bread-cat-' . $cat_id . ' bread-cat-' . $cat_nicename . '" href="' . $cat_link . '" title="' . $cat_name . '">' . $cat_name . '</a></span>';
						$html .= '<span class="separator"> ' . $separator . ' </span>';
						$html .= '<span class="item-current item-' . $post->ID . '"><span class="bread-current bread-' . $post->ID . '" title="' . get_the_title() . '">' . get_the_title() . '</span></span>';
					} else {
						$html .= '<span class="item-current item-' . $post->ID . '"><span class="bread-current bread-' . $post->ID . '" title="' . get_the_title() . '">' . get_the_title() . '</span></span>';
					}
				} elseif ( is_category() ) {
					// Category page
					$html .= '<span class="item-current item-cat"><span class="bread-current bread-cat">' . single_cat_title( '', false ) . '</span></span>';
				} elseif ( is_page() ) {
					// Standard page
					if ( $post->post_parent ) {
						// If child page, get parents
						$anc = get_post_ancestors( $post->ID );

						// Get parents in the right order
						$anc = array_reverse( $anc );

						// Parent page loop
						if ( ! isset( $parents ) ) {
							$parents = null;
						}
						foreach ( $anc as $ancestor ) {
							$parents .= '<span class="item-parent item-parent-' . $ancestor . '"><a class="bread-parent bread-parent-' . $ancestor . '" href="' . get_permalink( $ancestor ) . '" title="' . get_the_title( $ancestor ) . '">' . get_the_title( $ancestor ) . '</a></span>';
							$parents .= '<span class="separator separator-' . $ancestor . '"> ' . $separator . ' </span>';
						}

						// Display parent pages
						$html .= $parents;

						// Current page
						$html .= '<span class="item-current item-' . $post->ID . '"><span title="' . get_the_title() . '"> ' . get_the_title() . '</span></span>';
					} else {
						// Just display current page if not parents
						$html .= '<span class="item-current item-' . $post->ID . '"><span class="bread-current bread-' . $post->ID . '"> ' . get_the_title() . '</span></span>';
					}
				} elseif ( is_tag() ) {
					// Tag page
					// Get tag information
					$term_id       = get_query_var( 'tag_id' );
					$taxonomy      = 'post_tag';
					$args          = 'include=' . $term_id;
					$terms         = get_terms( $taxonomy, $args );
					$get_term_id   = $terms[0]->term_id;
					$get_term_slug = $terms[0]->slug;
					$get_term_name = $terms[0]->name;

					// Display the tag name
					$html .= '<span class="item-current item-tag-' . $get_term_id . ' item-tag-' . $get_term_slug . '"><span class="bread-current bread-tag-' . $get_term_id . ' bread-tag-' . $get_term_slug . '">' . $get_term_name . '</span></span>';
				} elseif ( is_day() ) {
					// Day archive

					// Year link
					$html .= '<span class="item-year item-year-' . get_the_time( 'Y' ) . '"><a class="bread-year bread-year-' . get_the_time( 'Y' ) . '" href="' . get_year_link( get_the_time( 'Y' ) ) . '" title="' . get_the_time( 'Y' ) . '">' . get_the_time( 'Y' ) . '</a></span>';
					$html .= '<span class="separator separator-' . get_the_time( 'Y' ) . '"> ' . $separator . ' </span>';

					// Month link
					$html .= '<span class="item-month item-month-' . get_the_time( 'm' ) . '"><a class="bread-month bread-month-' . get_the_time( 'm' ) . '" href="' . get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ) . '" title="' . get_the_time( 'M' ) . '">' . get_the_time( 'M' ) . '</a></span>';
					$html .= '<span class="separator separator-' . get_the_time( 'm' ) . '"> ' . $separator . ' </span>';

					// Day display
					$html .= '<span class="item-current item-' . get_the_time( 'j' ) . '"><span class="bread-current bread-' . get_the_time( 'j' ) . '"> ' . get_the_time( 'j' ) . '</span></span>';
				} elseif ( is_month() ) {
					// Month Archive

					// Year link
					$html .= '<span class="item-year item-year-' . get_the_time( 'Y' ) . '"><a class="bread-year bread-year-' . get_the_time( 'Y' ) . '" href="' . get_year_link( get_the_time( 'Y' ) ) . '" title="' . get_the_time( 'Y' ) . '">' . get_the_time( 'Y' ) . '</a></span>';
					$html .= '<span class="separator separator-' . get_the_time( 'Y' ) . '"> ' . $separator . ' </span>';

					// Month display
					$html .= '<span class="item-month item-month-' . get_the_time( 'm' ) . '"><span class="bread-month bread-month-' . get_the_time( 'm' ) . '" title="' . get_the_time( 'M' ) . '">' . get_the_time( 'M' ) . '</span></span>';
				} elseif ( is_year() ) {
					// Display year archive
					$html .= '<span class="item-current item-current-' . get_the_time( 'Y' ) . '"><span class="bread-current bread-current-' . get_the_time( 'Y' ) . '" title="' . get_the_time( 'Y' ) . '">' . get_the_time( 'Y' ) . '</span></span>';
				} elseif ( is_author() ) {
					// Auhor archive

					// Get the author information
					global $author;
					$userdata = get_userdata( $author );

					// Display author name
					$html .= '<span class="item-current item-current-' . $userdata->user_nicename . '"><span class="bread-current bread-current-' . $userdata->user_nicename . '" title="' . $userdata->display_name . '">' . $userdata->display_name . '</span></span>';
				} elseif ( get_query_var( 'paged' ) ) {
					// Paginated archives
					$html .= '<span class="item-current item-current-' . get_query_var( 'paged' ) . '"><span class="bread-current bread-current-' . get_query_var( 'paged' ) . '" title="Page ' . get_query_var( 'paged' ) . '">' . wp_sprintf( esc_html__( 'Page %s', 'kubelio' ), get_query_var( 'paged' ) ) . '</span></span>';
				} elseif ( is_search() ) {
					// Search results page
					$query = get_search_query( false );
					$html .= '<span class="item-current item-current-' . sanitize_html_class( $query ) . '"><span class="bread-current bread-current-' . sanitize_html_class( $query ) . '" title="' . esc_attr( sprintf( __( 'Search results for: %s', 'kubelio' ), $query ) ) . '">' . wp_sprintf( esc_html__( 'Search results for: %s', 'kubelio' ), esc_html( $query ) ) . '</span></span>';
				} elseif ( is_404() ) {
					// 404 page
					$html .= '<span>' . esc_html__( 'Error 404', 'kubelio' ) . '</span>';
				} elseif ( is_archive() && ! is_tax() ) {
					$html .= '<span class="item-current item-archive"><span class="bread-current bread-archive">' . post_type_archive_title( '', false ) . '</span></span>';
				} elseif ( is_archive() && is_tax() ) {
					// If post is a custom post type
					$post_type = get_post_type();

					// If it is a custom post type display name and link
					if ( $post_type != 'post' && isset( $post_type_object ) && $post_type_object instanceof \WP_Post_Type ) {
						$post_type_archive = get_post_type_archive_link( $post_type );

						if ( $post_type_archive ) {
							$html .= '<span class="item-cat item-custom-post-type-' . $post_type . '"><a class="bread-cat bread-custom-post-type-' . $post_type . '" href="' . esc_url( $post_type_archive ) . '" title="' . esc_attr( $post_type_object->labels->name ) . '">' . esc_html( $post_type_object->labels->name ) . '</a></span>';
							$html .= '<span class="separator"> ' . esc_html( $separator ) . ' </span>';
						}
					}

					$custom_tax_name = get_queried_object()->name;
					$html           .= '<span class="item-current item-archive"><span class="bread-current bread-archive">' . $custom_tax_name . '</span></span>';
				}
			}
		}
		$html .= '</div></div>';
		$html .= ob_get_clean();

		return $html;
	}
}

/**
 * Get pagination
 *
 * @param array $args
 * @return string
 */
function rt_get_pagination( $args = array() ) {

	// Build query
	$args = wp_parse_args(
		$args,
		array(
			'target'           => 'post_loops',
			'post_type'        => 'post',
			'post_status'      => 'publish',
			'posts_per_page'   => 4,
			'total_page'       => 3,
			'post_total'       => 12,
			'pagination_style' => 'number',
			'pagination_label' => esc_html__( 'Load More', 'kubelio' ),
		)
	);

	// Generate HTML
	$html = '';

	if ( $args['total_page'] > 1 ) {

		// Pagination load more
		if ( $args['pagination_style'] == 'loadmore' ) {

			unset( $args['paged'] );

			$options = htmlspecialchars( json_encode( $args ) );

			$html  = '<div class="rt-pagination is-type-loadmore mt-60">';
			$html .= '<button 
                        data-target="' . $args['target'] . '"
                        data-current-page="1"
                        data-total-page="' . $args['total_page'] . '"
                        data-post-perpage="' . $args['posts_per_page'] . '"
                        data-post-total="' . $args['post_total'] . '"
                        data-setting= "' . $options . '"
                        arial-label="' . $args['pagination_label'] . '"
                        class="js-loop-loadmore rt-btn is-style-outline">' . $args['pagination_label'] . '</button>';
			$html .= '</div>';
		}

		// Pagination number
		if ( $args['pagination_style'] == 'number' ) {
			$html  = '<div class="rt-pagination is-type-number mt-60">';
			$html .= paginate_links(
				wp_parse_args(
					$args,
					array(
						'prev_text' => '<i class="rt-icon"><svg xmlns="http://www.w3.org/2000/svg" width=1em" height="1em fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/></svg></i>',
						'next_text' => '<i class="rt-icon"><svg xmlns="http://www.w3.org/2000/svg" width="1em" height="1em" fill="currentColor" class="bi bi-arrow-right" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M1 8a.5.5 0 0 1 .5-.5h11.793l-3.147-3.146a.5.5 0 0 1 .708-.708l4 4a.5.5 0 0 1 0 .708l-4 4a.5.5 0 0 1-.708-.708L13.293 8.5H1.5A.5.5 0 0 1 1 8z"/></svg></i>',
						'current'   => ( is_front_page() && get_option( 'show_on_front' ) == 'page' ) ? max( 1, get_query_var( 'page' ) ) : max( 1, get_query_var( 'paged' ) ),
					)
				)
			);
			$html .= '</div>';
		}
	}

	return $html;
}

/**
 * Check woocommerce page
 *
 * @param string $page
 * @return bool
 */
function rt_is_woocommerce( $pages = array() ) {

	if ( ! class_exists( 'WooCommerce' ) ) {
		return false;
	}

	// Normalize to array
	$pages = (array) $pages;

	/**
	 * If parameter is empty:
	 *   → return true if current page is WooCommerce page
	 */
	if ( empty( $pages ) ) {
		return (
			is_woocommerce()
			|| is_shop()
			|| is_product()
			|| is_product_taxonomy()
			|| is_cart()
			|| is_checkout()
			|| is_account_page()
			|| is_wc_endpoint_url()
		);
	}

	// List of WooCommerce conditions
	$conditions = array(
		'pages'        => is_woocommerce() ?? false,
		'shop'         => is_shop() || is_tax( get_object_taxonomies( 'product' ) ) ?? false,
		'product'      => is_product() ?? false,
		'account'      => is_account_page() ?? false,
		'cart'         => is_cart() ?? false,
		'checkout'     => is_checkout() ?? false,
		'endpoint_url' => is_wc_endpoint_url() ?? false,
		'category'     => is_product_category() ?? false,
		'tag'          => is_product_tag() ?? false,
		'archive'      => is_product_taxonomy() || is_product_category() || is_product_tag() || is_shop() ?? false,
		'any'          => true,
		'thank_page'   => is_wc_endpoint_url( 'order-received' ) ?? false,
	);

	// Loop through conditions
	foreach ( $pages as $page ) {
		if ( isset( $conditions[ $page ] ) && $conditions[ $page ] ) {
			return true;
		}
	}

	return false;
}


/**
 * Get site copyright
 *
 * @return string
 */
function rt_get_site_copyright() {
	$theme = rt_get_theme( 'product-name' );

	$text_option = get_theme_mod( 'footer_bottom_text', '@Copyright ' . get_bloginfo( 'name' ) . '. All Rights Reserved' );
	$text_format = str_replace( array( '{{site_name}}', '{{year}}' ), array( get_bloginfo( 'name' ), date( 'Y' ) ), $text_option );

	$html = "<span class='site-copyright'>{$text_format}</span>";

	return $html;
}

/**
 * Generate social media icon link
 */
function rt_get_social_media( $args = '' ) {
	$args = wp_parse_args(
		$args,
		array(
			'type' => 'default',
			'size' => 'inherit',
		)
	);

	$social_links = array(
		'facebook'  => get_theme_mod( 'socmed_facebook', '#' ),
		'instagram' => get_theme_mod( 'socmed_instagram', '#' ),
		'youtube'   => get_theme_mod( 'socmed_youtube', '#' ),
		'twitter'   => get_theme_mod( 'socmed_twitter', '#' ),
		'pinterest' => get_theme_mod( 'socmed_pinterest', '#' ),
		'tiktok'    => get_theme_mod( 'socmed_tiktok', '#' ),
		'github'    => get_theme_mod( 'socmed_github', '' ),
		'gitlab'    => get_theme_mod( 'socmed_gitlab', '' ),
		'behance'   => get_theme_mod( 'socmed_behance', '' ),
		'dribbble'  => get_theme_mod( 'socmed_dribbble', '' ),
		'linkedin'  => get_theme_mod( 'socmed_linkedin', '' ),
	);

	if ( array_filter( $social_links ) ) {
		$html = '<ul class="wp-block-social-links is-layout-flex is-style-' . esc_attr( $args['type'] ) . ' has-' . esc_attr( $args['size'] ) . '-icon-size gap-30">';

		foreach ( $social_links as $platform => $link ) {
			if ( $link ) {
				$html .= '<li class="wp-social-link wp-social-link-' . esc_attr( $platform ) . ' wp-block-social-link">';
				$html .= '<a class="wp-block-social-link-anchor" href="' . esc_url( $link ) . '" target="_blank" rel="noopener noreferrer nofollow" aria-label="' . sprintf( esc_html__( 'follow on %s', 'kubelio' ), esc_attr( $platform ) ) . '">';

				switch ( $platform ) {
					case 'facebook':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-facebook" viewBox="0 0 16 16"><path d="M16 8.049c0-4.446-3.582-8.05-8-8.05C3.58 0-.002 3.603-.002 8.05c0 4.017 2.926 7.347 6.75 7.951v-5.625h-2.03V8.05H6.75V6.275c0-2.017 1.195-3.131 3.022-3.131.876 0 1.791.157 1.791.157v1.98h-1.009c-.993 0-1.303.621-1.303 1.258v1.51h2.218l-.354 2.326H9.25V16c3.824-.604 6.75-3.934 6.75-7.951"/></svg>';
						break;
					case 'instagram':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-instagram" viewBox="0 0 16 16"><path d="M8 0C5.829 0 5.556.01 4.703.048 3.85.088 3.269.222 2.76.42a3.9 3.9 0 0 0-1.417.923A3.9 3.9 0 0 0 .42 2.76C.222 3.268.087 3.85.048 4.7.01 5.555 0 5.827 0 8.001c0 2.172.01 2.444.048 3.297.04.852.174 1.433.372 1.942.205.526.478.972.923 1.417.444.445.89.719 1.416.923.51.198 1.09.333 1.942.372C5.555 15.99 5.827 16 8 16s2.444-.01 3.298-.048c.851-.04 1.434-.174 1.943-.372a3.9 3.9 0 0 0 1.416-.923c.445-.445.718-.891.923-1.417.197-.509.332-1.09.372-1.942C15.99 10.445 16 10.173 16 8s-.01-2.445-.048-3.299c-.04-.851-.175-1.433-.372-1.941a3.9 3.9 0 0 0-.923-1.417A3.9 3.9 0 0 0 13.24.42c-.51-.198-1.092-.333-1.943-.372C10.443.01 10.172 0 7.998 0zm-.717 1.442h.718c2.136 0 2.389.007 3.232.046.78.035 1.204.166 1.486.275.373.145.64.319.92.599s.453.546.598.92c.11.281.24.705.275 1.485.039.843.047 1.096.047 3.231s-.008 2.389-.047 3.232c-.035.78-.166 1.203-.275 1.485a2.5 2.5 0 0 1-.599.919c-.28.28-.546.453-.92.598-.28.11-.704.24-1.485.276-.843.038-1.096.047-3.232.047s-2.39-.009-3.233-.047c-.78-.036-1.203-.166-1.485-.276a2.5 2.5 0 0 1-.92-.598 2.5 2.5 0 0 1-.6-.92c-.109-.281-.24-.705-.275-1.485-.038-.843-.046-1.096-.046-3.233s.008-2.388.046-3.231c.036-.78.166-1.204.276-1.486.145-.373.319-.64.599-.92s.546-.453.92-.598c.282-.11.705-.24 1.485-.276.738-.034 1.024-.044 2.515-.045zm4.988 1.328a.96.96 0 1 0 0 1.92.96.96 0 0 0 0-1.92m-4.27 1.122a4.109 4.109 0 1 0 0 8.217 4.109 4.109 0 0 0 0-8.217m0 1.441a2.667 2.667 0 1 1 0 5.334 2.667 2.667 0 0 1 0-5.334"/></svg>';
						break;
					case 'youtube':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-youtube" viewBox="0 0 16 16"><path d="M8.051 1.999h.089c.822.003 4.987.033 6.11.335a2.01 2.01 0 0 1 1.415 1.42c.101.38.172.883.22 1.402l.01.104.022.26.008.104c.065.914.073 1.77.074 1.957v.075c-.001.194-.01 1.108-.082 2.06l-.008.105-.009.104c-.05.572-.124 1.14-.235 1.558a2.01 2.01 0 0 1-1.415 1.42c-1.16.312-5.569.334-6.18.335h-.142c-.309 0-1.587-.006-2.927-.052l-.17-.006-.087-.004-.171-.007-.171-.007c-1.11-.049-2.167-.128-2.654-.26a2.01 2.01 0 0 1-1.415-1.419c-.111-.417-.185-.986-.235-1.558L.09 9.82l-.008-.104A31 31 0 0 1 0 7.68v-.123c.002-.215.01-.958.064-1.778l.007-.103.003-.052.008-.104.022-.26.01-.104c.048-.519.119-1.023.22-1.402a2.01 2.01 0 0 1 1.415-1.42c.487-.13 1.544-.21 2.654-.26l.17-.007.172-.006.086-.003.171-.007A100 100 0 0 1 7.858 2zM6.4 5.209v4.818l4.157-2.408z"/></svg>';
						break;
					case 'twitter':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-twitter-x" viewBox="0 0 16 16"><path d="M12.6.75h2.454l-5.36 6.142L16 15.25h-4.937l-3.867-5.07-4.425 5.07H.316l5.733-6.57L0 .75h5.063l3.495 4.633L12.601.75Zm-.86 13.028h1.36L4.323 2.145H2.865z"/></svg>';
						break;
					case 'pinterest':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-pinterest" viewBox="0 0 16 16"><path d="M8 0a8 8 0 0 0-2.915 15.452c-.07-.633-.134-1.606.027-2.297.146-.625.938-3.977.938-3.977s-.239-.479-.239-1.187c0-1.113.645-1.943 1.448-1.943.682 0 1.012.512 1.012 1.127 0 .686-.437 1.712-.663 2.663-.188.796.4 1.446 1.185 1.446 1.422 0 2.515-1.5 2.515-3.664 0-1.915-1.377-3.254-3.342-3.254-2.276 0-3.612 1.707-3.612 3.471 0 .688.265 1.425.595 1.826a.24.24 0 0 1 .056.23c-.061.252-.196.796-.222.907-.035.146-.116.177-.268.107-1-.465-1.624-1.926-1.624-3.1 0-2.523 1.834-4.84 5.286-4.84 2.775 0 4.932 1.977 4.932 4.62 0 2.757-1.739 4.976-4.151 4.976-.811 0-1.573-.421-1.834-.919l-.498 1.902c-.181.695-.669 1.566-.995 2.097A8 8 0 1 0 8 0"/></svg>';
						break;
					case 'tiktok':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-tiktok" viewBox="0 0 16 16"><path d="M9 0h1.98c.144.715.54 1.617 1.235 2.512C12.895 3.389 13.797 4 15 4v2c-1.753 0-3.07-.814-4-1.829V11a5 5 0 1 1-5-5v2a3 3 0 1 0 3 3z"/></svg>';
						break;
					case 'github':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-github" viewBox="0 0 16 16"><path d="M8 0C3.58 0 0 3.58 0 8c0 3.54 2.29 6.53 5.47 7.59.4.07.55-.17.55-.38 0-.19-.01-.82-.01-1.49-2.01.37-2.53-.49-2.69-.94-.09-.23-.48-.94-.82-1.13-.28-.15-.68-.52-.01-.53.63-.01 1.08.58 1.23.82.72 1.21 1.87.87 2.33.66.07-.52.28-.87.51-1.07-1.78-.2-3.64-.89-3.64-3.95 0-.87.31-1.59.82-2.15-.08-.2-.36-1.02.08-2.12 0 0 .67-.21 2.2.82.64-.18 1.32-.27 2-.27s1.36.09 2 .27c1.53-1.04 2.2-.82 2.2-.82.44 1.1.16 1.92.08 2.12.51.56.82 1.27.82 2.15 0 3.07-1.87 3.75-3.65 3.95.29.25.54.73.54 1.48 0 1.07-.01 1.93-.01 2.2 0 .21.15.46.55.38A8.01 8.01 0 0 0 16 8c0-4.42-3.58-8-8-8"/></svg>';
						break;
					case 'gitlab':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-gitlab" viewBox="0 0 16 16"><path d="m15.734 6.1-.022-.058L13.534.358a.57.57 0 0 0-.563-.356.6.6 0 0 0-.328.122.6.6 0 0 0-.193.294l-1.47 4.499H5.025l-1.47-4.5A.572.572 0 0 0 2.47.358L.289 6.04l-.022.057A4.044 4.044 0 0 0 1.61 10.77l.007.006.02.014 3.318 2.485 1.64 1.242 1 .755a.67.67 0 0 0 .814 0l1-.755 1.64-1.242 3.338-2.5.009-.007a4.05 4.05 0 0 0 1.34-4.668Z"/></svg>';
						break;
					case 'behance':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-behance" viewBox="0 0 16 16"><path d="M4.654 3c.461 0 .887.035 1.278.14.39.07.711.216.996.391s.497.426.641.747c.14.32.216.711.216 1.137 0 .496-.106.922-.356 1.242-.215.32-.566.606-.997.817.606.176 1.067.496 1.348.922s.461.957.461 1.563c0 .496-.105.922-.285 1.278a2.3 2.3 0 0 1-.782.887c-.32.215-.711.39-1.137.496a5.3 5.3 0 0 1-1.278.176L0 12.803V3zm-.285 3.978c.39 0 .71-.105.957-.285.246-.18.355-.497.355-.887 0-.216-.035-.426-.105-.567a1 1 0 0 0-.32-.355 1.8 1.8 0 0 0-.461-.176c-.176-.035-.356-.035-.567-.035H2.17v2.31c0-.005 2.2-.005 2.2-.005zm.105 4.193c.215 0 .426-.035.606-.07.176-.035.356-.106.496-.216s.25-.215.356-.39c.07-.176.14-.391.14-.641 0-.496-.14-.852-.426-1.102-.285-.215-.676-.32-1.137-.32H2.17v2.734h2.305zm6.858-.035q.428.427 1.278.426c.39 0 .746-.106 1.032-.286q.426-.32.53-.64h1.74c-.286.851-.712 1.457-1.278 1.848-.566.355-1.243.566-2.06.566a4.1 4.1 0 0 1-1.527-.285 2.8 2.8 0 0 1-1.137-.782 2.85 2.85 0 0 1-.712-1.172c-.175-.461-.25-.957-.25-1.528 0-.531.07-1.032.25-1.493.18-.46.426-.852.747-1.207.32-.32.711-.606 1.137-.782a4 4 0 0 1 1.493-.285c.606 0 1.137.105 1.598.355.46.25.817.532 1.102.958.285.39.496.851.641 1.348.07.496.105.996.07 1.563h-5.15c0 .58.21 1.11.496 1.396m2.24-3.732c-.25-.25-.642-.391-1.103-.391-.32 0-.566.07-.781.176s-.356.25-.496.39a.96.96 0 0 0-.25.497c-.036.175-.07.32-.07.46h3.196c-.07-.526-.25-.882-.497-1.132zm-3.127-3.728h3.978v.957h-3.978z"/></svg>';
						break;
					case 'dribbble':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-dribbble" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M8 0C3.584 0 0 3.584 0 8s3.584 8 8 8c4.408 0 8-3.584 8-8s-3.592-8-8-8m5.284 3.688a6.8 6.8 0 0 1 1.545 4.251c-.226-.043-2.482-.503-4.755-.217-.052-.112-.096-.234-.148-.355-.139-.33-.295-.668-.451-.99 2.516-1.023 3.662-2.498 3.81-2.69zM8 1.18c1.735 0 3.323.65 4.53 1.718-.122.174-1.155 1.553-3.584 2.464-1.12-2.056-2.36-3.74-2.551-4A7 7 0 0 1 8 1.18m-2.907.642A43 43 0 0 1 7.627 5.77c-3.193.85-6.013.833-6.317.833a6.87 6.87 0 0 1 3.783-4.78zM1.163 8.01V7.8c.295.01 3.61.053 7.02-.971.199.381.381.772.555 1.162l-.27.078c-3.522 1.137-5.396 4.243-5.553 4.504a6.82 6.82 0 0 1-1.752-4.564zM8 14.837a6.8 6.8 0 0 1-4.19-1.44c.12-.252 1.509-2.924 5.361-4.269.018-.009.026-.009.044-.017a28.3 28.3 0 0 1 1.457 5.18A6.7 6.7 0 0 1 8 14.837m3.81-1.171c-.07-.417-.435-2.412-1.328-4.868 2.143-.338 4.017.217 4.251.295a6.77 6.77 0 0 1-2.924 4.573z"/></svg>';
						break;
					case 'linkedin':
						$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-linkedin" viewBox="0 0 16 16"><path d="M0 1.146C0 .513.526 0 1.175 0h13.65C15.474 0 16 .513 16 1.146v13.708c0 .633-.526 1.146-1.175 1.146H1.175C.526 16 0 15.487 0 14.854zm4.943 12.248V6.169H2.542v7.225zm-1.2-8.212c.837 0 1.358-.554 1.358-1.248-.015-.709-.52-1.248-1.342-1.248S2.4 3.226 2.4 3.934c0 .694.521 1.248 1.327 1.248zm4.908 8.212V9.359c0-.216.016-.432.08-.586.173-.431.568-.878 1.232-.878.869 0 1.216.662 1.216 1.634v3.865h2.401V9.25c0-2.22-1.184-3.252-2.764-3.252-1.274 0-1.845.7-2.165 1.193v.025h-.016l.016-.025V6.169h-2.4c.03.678 0 7.225 0 7.225z"/></svg>';
						break;
				}

				$html .= '</a></li>';
			}
		}

		$html .= '</ul>';
		return $html;
	}
}

/**
 * Get wishlist count
 *
 * value from Kubelio Booster plugin option
 */
function rt_get_wishlist_count() {
	if ( rt_fs()->can_use_premium_code() ) {
		if ( function_exists( 'kubelio_companion_get_option' ) && function_exists( 'kubelio_companion_wishlist_get_count' ) ) {
			return kubelio_companion_wishlist_get_count();
		}
	}

	return 0;
}

/**
 * Check if wishlist is enabled
 */
function rt_is_wishlist_enabled() {
	if ( rt_fs()->can_use_premium_code() ) {
		if ( function_exists( 'kubelio_companion_get_option' ) ) {
			return kubelio_companion_get_option( 'wishlist', 'yes' ) ? true : false;
		}
	}

	return false;
}
