<?php

/**
 * Theme setup and core functionality
 *
 * This file contains functions that handle the basic setup of the theme,
 * including theme supports, default settings, and core functionality
 * that needs to be loaded early in the WordPress lifecycle.
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function rt_setup_theme() {
	add_theme_support( 'responsive-embeds' );
	add_theme_support( 'post-thumbnails' );
	add_theme_support( 'title-tag' );
	add_theme_support( 'block-template-parts' );

	// Add theme support selective refresh for customizer
	add_theme_support( 'customize-selective-refresh-widgets' );

	// Register theme menu
	register_nav_menus(
		array(
			'primary' => 'Primary Menu',
			'footer'  => 'Footer Menu',
			'mobile'  => 'Mobile Menu',
		)
	);

	// Add custom image size
	add_image_size( 'featured_medium', 420, 999, false );

	// Hook after setup theme
	do_action( 'retheme_after_setup_theme' );
}
add_action( 'after_setup_theme', 'rt_setup_theme' );

/**
 * Register widget areas (sidebars)
 *
 * Registers the theme's widget areas including sidebars, footer widgets,
 * and any other widgetized areas used throughout the theme.
 */
function rt_setup_register_widget() {

	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'kubelio' ),
			'id'            => 'retheme_sidebar',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-aside %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-block-heading"><h3 class="rt-block-heading__title has-medium-font-size">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 1', 'kubelio' ),
			'id'            => 'retheme_footer_1',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 2', 'kubelio' ),
			'id'            => 'retheme_footer_2',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 3', 'kubelio' ),
			'id'            => 'retheme_footer_3',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 4', 'kubelio' ),
			'id'            => 'retheme_footer_4',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 5', 'kubelio' ),
			'id'            => 'retheme_footer_5',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 6', 'kubelio' ),
			'id'            => 'retheme_footer_6',
			'before_widget' => '<div id="%1$s" class="rt-widget is-widget-footer is-layout-flow %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="rt-widget__header"><h3 class="rt-widget__title">',
			'after_title'   => '</h3></div>',
		)
	);
}
add_action( 'widgets_init', 'rt_setup_register_widget' );


/**
 * Get default theme variable
 *
 * Set default theme variable from theme.json
 *
 * @param string $value
 * @return string
 */
function rt_get_theme( $value = '' ) {
	static $default = null;

	if ( $default === null ) {
		$theme_colors    = array();
		$custom_settings = array();

		// Helper function for merge color palettes
		$merge_color_palettes = function ( $parent_colors, $child_colors ) {
			$merged     = array();
			$all_colors = array_merge( $parent_colors, $child_colors );

			// Use slug as key for override
			foreach ( $all_colors as $color ) {
				if ( isset( $color['slug'] ) && isset( $color['color'] ) ) {
					$merged[ $color['slug'] ] = $color['color'];
				}
			}

			return $merged;
		};

		// Get parent theme.json
		$parent_json_file = get_template_directory() . '/theme.json';
		$parent_colors    = array();
		$parent_custom    = array();

		if ( file_exists( $parent_json_file ) ) {
			$parent_data   = json_decode( file_get_contents( $parent_json_file ), true );
			$parent_colors = $parent_data['settings']['color']['palette'] ?? array();
			$parent_custom = $parent_data['settings']['custom'] ?? array();
		}

		// Get child theme.json (if different from parent)
		$child_colors = array();
		$child_custom = array();

		if ( get_template_directory() !== get_stylesheet_directory() ) {
			$child_json_file = get_stylesheet_directory() . '/theme.json';

			if ( file_exists( $child_json_file ) ) {
				$child_data   = json_decode( file_get_contents( $child_json_file ), true );
				$child_colors = $child_data['settings']['color']['palette'] ?? array();
				$child_custom = $child_data['settings']['custom'] ?? array();
			}
		}

		// Merge colors (child override parent)
		$theme_colors = $merge_color_palettes( $parent_colors, $child_colors );

		// Merge custom settings (child override parent)
		$custom_settings = array_replace_recursive( $parent_custom, $child_custom );

		$default = array(
			'product-name'    => RT_THEME_NAME,
			'product-slug'    => RT_THEME_SLUG,
			'product-version' => RT_THEME_VERSION,
			'product-docs'    => RT_THEME_DOC,
			'product-url'     => RT_THEME_URL,
			'extends'         => array(),
			'typography'      => array(
				'heading' => array(
					'font-family' => 'Inter',
					'variant'     => '600',
				),
				'body'    => array(
					'font-family' => 'Inter',
					'variant'     => 'regular',
				),
				'menu'    => array(
					'font-family' => 'Inter',
					'variant'     => '700',
				),
			),
			'button'          => array(
				'primary' => array(
					'color'      => $custom_settings['button']['color'] ?? '',
					'background' => $custom_settings['button']['background'] ?? '',
				),
				'action'  => array(
					'color'      => $custom_settings['button-action']['color'] ?? '',
					'background' => $custom_settings['button-action']['background'] ?? '',
				),
			),
			'color'           => $theme_colors,
		);
	}

	return ! empty( $default[ $value ] ) ? $default[ $value ] : '';
}

/*
 * Add theme color brand on search board mobile browser
 *
 * @version 1.0.0
 */
function rt_setup_theme_color() {

	$color = get_theme_mod( 'brand_color_browser' );

	if ( ! empty( $color ) ) {
		echo '<meta name="theme-color" content="' . $color . '">';
	}
}
add_action( 'wp_head', 'rt_setup_theme_color' );

/**
 * Setup Merlin config
 *
 * Merlin config and set text string
 *
 * @version 1.0.0
 */
function rt_setup_merlin_config() {

	// Only initialize Merlin if the class exists
	if ( ! class_exists( 'Merlin' ) ) {
		return;
	}

	new Merlin(
		$config  = array(
			'directory'             => 'vendor/merlin',
			'merlin_url'            => 'merlin',
			'parent_slug'           => '',
			'capability'            => 'manage_options',
			'dev_mode'              => true,
			'license_step'          => false,
			'license_required'      => false,
			'license_help_url'      => '',
			'edd_remote_api_url'    => '',
			'edd_item_name'         => '',
			'edd_theme_slug'        => '',
			'ready_big_button_url'  => site_url(),
			'redirect_switch_theme' => false,
		),
		$strings = array(
			'admin-menu'               => esc_html__( 'Theme Setup', 'kubelio' ),
			'title%s%s%s%s'            => esc_html__( '%1$s%2$s Themes &lsaquo; Theme Setup: %3$s%4$s', 'kubelio' ),
			'return-to-dashboard'      => esc_html__( 'Return to Dashboard', 'kubelio' ),
			'ignore'                   => esc_html__( 'Disable Theme Setup', 'kubelio' ),
			'btn-skip'                 => esc_html__( 'Skip', 'kubelio' ),
			'btn-next'                 => esc_html__( 'Next', 'kubelio' ),
			'btn-start'                => esc_html__( 'Start', 'kubelio' ),
			'btn-no'                   => esc_html__( 'Cancel', 'kubelio' ),
			'btn-plugins-install'      => esc_html__( 'Install', 'kubelio' ),
			'btn-child-install'        => esc_html__( 'Install', 'kubelio' ),
			'btn-content-install'      => esc_html__( 'Install', 'kubelio' ),
			'btn-import'               => esc_html__( 'Import', 'kubelio' ),
			'btn-license-activate'     => esc_html__( 'Activate', 'kubelio' ),
			'btn-license-skip'         => esc_html__( 'Later', 'kubelio' ),
			'license-header%s'         => esc_html__( 'Activate %s', 'kubelio' ),
			'license-header-success%s' => esc_html__( '%s has been activated', 'kubelio' ),
			/* translators: Theme Name */
			'license%s'                => esc_html__( 'Enter the license key to unlock all premium features, updates, and support from Kubelio Studio', 'kubelio' ),
			'license-label'            => esc_html__( 'License Key', 'kubelio' ),
			'license-success%s'        => esc_html__( 'Congratulations! The theme has been activated. Let’s move to the next step.', 'kubelio' ),
			'license-json-success%s'   => esc_html__( 'Congratulations! The theme has been activated. Let’s move to the next step.', 'kubelio' ),
			'license-tooltip'          => esc_html__( 'Need help?', 'kubelio' ),
			/* translators: Theme Name */
			'welcome-header%s'         => esc_html__( 'Welcome to %s', 'kubelio' ),
			'welcome-header-success%s' => esc_html__( 'Hi. Welcome back', 'kubelio' ),
			'welcome%s'                => esc_html__( 'This Theme Setup feature will guide you through configuring the theme, installing plugins, and importing content.', 'kubelio' ),
			'welcome-success%s'        => esc_html__( 'You may have already run this Theme Setup before. If you want to continue, please click the "Start" button below.', 'kubelio' ),
			'child-header'             => esc_html__( 'Install Child Theme', 'kubelio' ),
			'child-header-success'     => esc_html__( 'You\'re good to go!', 'kubelio' ),
			'child'                    => esc_html__( 'Activate the child theme so you can easily make changes without affecting the main theme.', 'kubelio' ),
			'child-success%s'          => esc_html__( 'Your child theme has been successfully installed and activated.', 'kubelio' ),
			'child-action-link'        => esc_html__( 'Learn more about child themes', 'kubelio' ),
			'child-json-success%s'     => esc_html__( 'Your child theme has been installed and activated.', 'kubelio' ),
			'child-json-already%s'     => esc_html__( 'Your child theme has already been created and is now active.', 'kubelio' ),
			'plugins-header'           => esc_html__( 'Install Plugins', 'kubelio' ),
			'plugins-header-success'   => esc_html__( 'Awesome', 'kubelio' ),
			'plugins'                  => esc_html__( 'Let’s install some essential plugins to build your website.', 'kubelio' ),
			'plugins-success%s'        => esc_html__( 'All required plugins have been installed and updated.', 'kubelio' ),
			'plugins-action-link'      => esc_html__( 'Advanced', 'kubelio' ),
			'import-header'            => esc_html__( 'Import Content', 'kubelio' ),
			'import'                   => esc_html__( 'Run the import process! Get the look and features just like the demo site.', 'kubelio' ),
			'import-action-link'       => esc_html__( 'Advanced', 'kubelio' ),
			'ready-header'             => esc_html__( 'Installation is complete. Have fun!', 'kubelio' ),
			'ready%s'                  => esc_html__( 'Your theme has been set up. Enjoy your new theme developed by Kubelio Studio.', 'kubelio' ),
			'ready-action-link'        => esc_html__( 'Extra', 'kubelio' ),
			'ready-big-button'         => esc_html__( 'Visit Website', 'kubelio' ),
			'ready-link-2'             => sprintf( '<a href="%1$s" target="_blank">%2$s</a>', RT_THEME_DOC, esc_html__( 'Theme Tutorial', 'kubelio' ) ),
			'ready-link-3'             => sprintf( '<a href="%1$s" target="_blank">%2$s</a>', 'https://kubelio.com/support/', esc_html__( 'Contact Theme Support', 'kubelio' ) ),
		),
	);
}
add_action( 'after_setup_theme', 'rt_setup_merlin_config' );

/**
 * Skip Merlin generate Child Theme
 */
add_filter( 'wp_ajax_merlin_child_theme', '__return_false' );

/**
 * Check for license updates.
 */
function rt_setup_update_checker() {
	if ( function_exists( 'rt_fs' ) && ! rt_fs()->can_use_premium_code() && class_exists( 'YahnisElsts\PluginUpdateChecker\v5\PucFactory' ) ) {
		$plugin_update_url = 'https://kubelio.com/wp-content/uploads/kubelio/updated/kubelio.json';
		$my_update_checker = YahnisElsts\PluginUpdateChecker\v5\PucFactory::buildUpdateChecker( $plugin_update_url, get_template_directory(), 'kubelio' );
	}
}
add_action( 'init', 'rt_setup_update_checker' );

/**
 * Register the required plugins for this theme.
 */
function rt_import_required_plugins() {

	/*
	 * Array of plugin arrays. Required keys are name and slug.
	 * If the source is NOT from the .org repo, then source is also required.
	 */
	$plugins = array(
		array(
			'name'     => 'Kubelio Companion',
			'slug'     => 'kubelio-companion',
			'required' => true,
			'source'   => 'https://kubelio.com/wp-content/uploads/kubelio/files/kubelio-companion.zip',
		),
		array(
			'name'     => 'WooCommerce',
			'slug'     => 'woocommerce',
			'required' => true,
		),
		array(
			'name'     => 'Contact Form 7',
			'slug'     => 'contact-form-7',
			'required' => false,
		),
		array(
			'name'     => 'Limit Login Attempts Reloaded',
			'slug'     => 'limit-login-attempts-reloaded',
			'required' => false,
		),
		array(
			'name'     => 'Backup - UpdraftPlus',
			'slug'     => 'updraftplus',
			'required' => false,
		),
	);

	/*
	 * Array of configuration settings. Amend each line as needed.
	 *
	 * TGMPA will start providing localized text strings soon. If you already have translations of our standard
	 * strings available, please help us make TGMPA even better by giving us access to these translations or by
	 * sending in a pull-request with .po file(s) with the translations.
	 *
	 * Only uncomment the strings in the config array if you want to customize the strings.
	 */

	$config = array(
		'id'           => 'rt_tgmpa', // Unique ID for hashing notices for multiple instances of TGMPA.
		'default_path' => '', // Default absolute path to bundled plugins.
		'menu'         => 'tgmpa-install-plugins', // Menu slug.
		'parent_slug'  => 'theme-panel', // Parent menu slug.
		'capability'   => 'edit_theme_options', // Capability needed to view plugin install page, should be a capability associated with the parent menu used.
		'has_notices'  => true, // Show admin notices or not.
		'dismissable'  => true, // If false, a user cannot dismiss the nag message.
		'dismiss_msg'  => true, // If 'dismissable' is false, this message will be output at top of nag.
		'is_automatic' => false, // Automatically activate plugins after installation or not.
		'message'      => '', // Message to output right before the plugins table.
	);

	tgmpa( $plugins, $config );
}
add_action( 'tgmpa_register', 'rt_import_required_plugins' );

// Disable WooCommerce Wizard
add_filter( 'woocommerce_enable_setup_wizard', '__return_false' );

// Disable WooCommerce page creation
add_filter( 'woocommerce_create_pages', '__return_empty_array', 999 );

// Disable block widget editor
add_filter( 'use_widgets_block_editor', '__return_false' );
