<?php

/**
 * Demo content import and setup after import
 *
 * This file handles the import of demo content, widgets, and theme settings
 * when using the Merlin WP theme setup wizard. It includes functions to
 * configure the import process and set up the site after import.
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Remove default widgets during demo import
 *
 * This function is used to unset default widgets from widget areas
 * when importing demo content. Add your widget area to the array
 * to clear its default widgets during import.
 *
 * @param array $widget_areas Arguments for the sidebars_widgets widget areas.
 * @return array Modified array of widget area IDs to clear
 */
function rt_import_unset_default_widgets_args( $widget_areas ) {

	$widget_areas = array(
		'retheme_sidebar'      => array(),
		'retheme_shop_sidebar' => array(),
		'retheme_shop_filter'  => array(),
		'retheme_footer_1'     => array(),
		'retheme_footer_2'     => array(),
		'retheme_footer_3'     => array(),
		'retheme_footer_4'     => array(),
	);

	return $widget_areas;
}
add_filter( 'merlin_unset_default_widgets_args', 'rt_import_unset_default_widgets_args' );

/**
 * Demo import
 *
 * @param  array $import Array of demo import files.
 * @return array of demo import files.
 */
function rt_import_files_demo( $import ) {
	$key    = uniqid();
	$import = array(
		array(
			'import_file_name'           => 'Kubelio Main',
			'categories'                 => array( 'Personal' ),
			'import_file_url'            => "https://kubelio.com/wp-content/uploads/kubelio/import/kubelio-main.xml?key={$key}",
			'import_widget_file_url'     => "https://kubelio.com/wp-content/uploads/kubelio/import/kubelio-main.wie?key={$key}",
			'import_customizer_file_url' => "https://kubelio.com/wp-content/uploads/kubelio/import/kubelio-main.dat?key={$key}",
			'preview_url'                => "https://kubelio.com/wp-content/uploads/kubelio/import/kubelio-main.png?key={$key}",
		),
	);

	return $import;
}
add_filter( 'merlin_import_files', 'rt_import_files_demo' );

/**
 * Set default settings after demo import
 *
 * @param string $selected_import Selected import file.
 * @return void
 */
function rt_import_set_finish( $selected_import ) {

	// Assign menu locations
	$primary_menu = get_term_by( 'name', 'Primary Menu', 'nav_menu' );
	$mobile_menu  = get_term_by( 'name', 'Mobile Menu', 'nav_menu' );
	$footer_menu  = get_term_by( 'name', 'Footer Menu', 'nav_menu' );

	$menu_locations = array();

	if ( $primary_menu ) {
		$menu_locations['primary'] = $primary_menu->term_id;
	}
	if ( $mobile_menu ) {
		$menu_locations['mobile'] = $mobile_menu->term_id;
	}
	if ( $footer_menu ) {
		$menu_locations['footer'] = $footer_menu->term_id;
	}

	if ( ! empty( $menu_locations ) ) {
		set_theme_mod( 'nav_menu_locations', $menu_locations );
	}

	// Reading & general settings
	update_option( 'posts_per_page', 6 );
	update_option( 'show_on_front', 'page' );

	if ( $home = get_page_by_path( 'home' ) ) {
		update_option( 'page_on_front', $home->ID );
	}

	if ( $blog = get_page_by_path( 'blog' ) ) {
		update_option( 'page_for_posts', $blog->ID );
	}

	// Elementor & plugin defaults
	update_option( 'elementor_optimized_gutenberg_loading', 0 );

	$default_options = array(
		'updraft_interval'          => 'weekly',
		'updraft_retain'            => 3,
		'updraft_interval_database' => 'daily',
	);

	foreach ( $default_options as $key => $value ) {
		if ( false === get_option( $key ) ) {
			update_option( $key, $value );
		}
	}

	// Disable unwanted plugins
	$disable_plugins = array(
		'litespeed-cache/litespeed-cache.php',
		'speedycache/speedycache.php',
		'pagelayer/pagelayer.php',
		'wp-super-cache/wp-cache.php',
		'jetpack/jetpack.php',
	);

	deactivate_plugins( $disable_plugins, true );

	// Set default post to draft
	wp_update_post(
		array(
			'ID'          => 1,
			'post_status' => 'draft',
		)
	);

	// Replace demo URLs in theme mods
	if ( defined( 'RT_THEME_DEMO_ID' ) && defined( 'RT_THEME_DEMO_URL' ) ) {

		$site_url = get_site_url();
		$mods     = get_theme_mods();

		$demo_upload = RT_THEME_DEMO_URL . '/wp-content/uploads/sites/' . RT_THEME_DEMO_ID;
		$site_upload = $site_url . '/wp-content/uploads';

		if ( is_array( $mods ) ) {
			foreach ( $mods as $key => $value ) {
				if ( ! is_string( $value ) ) {
					continue;
				}

				$value = str_replace(
					array( $demo_upload, RT_THEME_DEMO_URL ),
					array( $site_upload, $site_url ),
					$value
				);

				set_theme_mod( $key, $value );
			}
		}
	}

	// Set demo imported
	set_theme_mod( 'demo_imported', true );

	// Disable comming soon
	update_option( 'woocommerce_coming_soon', 'no' );

	// Set WooCommerce Page
	update_option( 'woocommerce_shop_page_id', get_page_by_path( 'shop' )->ID );
	update_option( 'woocommerce_cart_page_id', get_page_by_path( 'cart' )->ID );
	update_option( 'woocommerce_checkout_page_id', get_page_by_path( 'checkout' )->ID );
	update_option( 'woocommerce_myaccount_page_id', get_page_by_path( 'my-account' )->ID );
}
add_action( 'merlin_after_all_import', 'rt_import_set_finish' );

/**
 * Set permalink after import
 */
function rt_import_set_permalink() {
	global $wp_rewrite;
	$wp_rewrite->set_permalink_structure( '/%postname%/' );
	$wp_rewrite->flush_rules();
}
add_action( 'retheme_merlin_import_final', 'rt_import_set_permalink' );
